# function for SIR model simulations

# runs a simulation of an SIR model
#
# S: susceptible
# I: infective
# R: removed
#
# beta is the infectivity parameter, scaled so that N=1
# gamma is the reciprocal of mean recovery time
# I0 is the initial infected fraction
# V is the initial immune fraction
# target is the infected fraction used as the end condition
#
# results is a matrix: columns are SIR, rows are 0:days
# 
# suggested default values
# beta = 0.5;
# gamma = 0.2;
# I0 = 0.01;
# V = 0;
# target = 0.001;
#
# by Glenn Ledder
# written 2020/07/14
# revised 2020/10/27
#
# direct comments to gledder@unl.edu

sir_sim <- function(beta,gamma,I0,V,target)
	{

	## DATA

	maxdays = 1000

	## FUNCTION FOR THE DIFFERENTIAL EQUATION

    	yprime <- function(yy)
		{
      	S = yy[1]
      	I = yy[2]
      	Sp = -beta*S*I
      	Ip = -Sp-gamma*I
      	yp = c(Sp,Ip)
		return(yp)
		}

	## INITIALIZATION

	# set up results data structure with Y=[S,I] and R

	results = matrix(0,nrow=maxdays+1,ncol=3)	
	S0 = 1-I0-V
	Y = c(S0,I0)
	R = V
	results[1,] = c(Y,R)

	oldI = I0

	## FUNCTION FOR rk4

      # y0 is a column vector of initial conditions at t
      # y is a column vector of values at t+dt

	rk4 <- function(dt,y0)
		{
		k1 = yprime(y0)
        	k2 = yprime(y0+0.5*dt*k1);
        	k3 = yprime(y0+0.5*dt*k2);
        	k4 = yprime(y0+dt*k3);
        	yy = y0+dt*(k1+2*k2+2*k3+k4)/6;
		return(yy)
		}

	## COMPUTATION

	for (t in 1:maxdays)
		{
		Y = rk4(1,Y)
		R = 1-sum(Y)
    		results[t+1,] = c(Y,R)
    		if (Y[2]>min(target,oldI))
			{
        		oldI = Y[2]
			} else {
	      	results = results[1:(t+1),]
        		break
			}
		}

	## OUTPUT

	return(results)

	## END

	}


